/* -*- mode: C; -*- */

/*
   Copyright (c) 1997-2002 MathEngine PLC

   $Name: t-stevet-RWSpre-030110 $

   Date: $Date: 2002/04/04 15:29:32 $ - Revision: $Revision: 1.46.2.2 $

   This software and its accompanying manuals have been developed
   by MathEngine PLC ("MathEngine") and the copyright and all other
   intellectual property rights in them belong to MathEngine. All
   rights conferred by law (including rights under international
   copyright conventions) are reserved to MathEngine. This software
   may also incorporate information which is confidential to
   MathEngine.

   Save to the extent permitted by law, or as otherwise expressly
   permitted by MathEngine, this software and the manuals must not
   be copied (in whole or in part), re-arranged, altered or adapted
   in any way without the prior written consent of the Company. In
   addition, the information contained in the software may not be
   disseminated without the prior written consent of MathEngine.

 */

/*
  Overview:

  Shows how small changes to BallHitsWall2 achieves dramatically
  different behavior. The first example is pre-compiled and
  ready to execute. The variations below are generated by
  performing small macro replacements directly in the program
  code. You need only perform these simple edits and recompile
  to obtain an executable.
*/


#include <math.h>
#include <stdlib.h>

#include <Mcd.h>
#include <McdPrimitives.h>
#include <Mst.h>
#include <MeViewer.h>

/* define to get cube behaviour */
#if 0
#define USE_CUBE
#endif
/* define to attach hinge joint to wall */

#define USE_HINGE

/*
  Global declarations.
*/

#define MAX_BODIES (10)
#define NCONTACTS   200

RRender *rc;

/*
  Dynamics world, collision space, and dynamics-collision bridge.
*/
MdtWorldID    world;
McdSpaceID    space;
MstBridgeID   bridge;
McdFrameworkID frame;

/*
  Physics reps.
*/
MdtBodyID box;
MdtBodyID ball;

/*
  Collision reps.
*/
McdGeometryID plane_prim, box_prim, box_prim1, ball_prim;
McdModelID groundCM;
McdModelID boxCM;
McdModelID ballCM;

/*
  Graphics reps.
*/
RGraphic *planeG;
RGraphic *boxG;
RGraphic *ballG;

#ifdef USE_HINGE
MdtHingeID hinge;
#endif

/*
  Misc.
*/

/* World gravity */
MeReal gravity[3] = { 0.0f, -5.0f, 0.0f };
/* Time step */
MeReal step = 0.01f;

/* Box mass */
MeReal boxMass = 5.0f;

/*  Ball mass */
MeReal ballMass = 3.0f;
/*  Ball radius */
MeReal radius = 0.5f;

/* Wall dimensions */
MeReal wallDims[3] = { 1.0f, 4.0f, 10.0f };
/* Wall position */
MeReal wallPos[3] = { 3.0f, 2.001f, 0.0f };
/* floor dimensions */
MeReal floorDims[3] = { 10.0f, 0.05f, 5.2f };

/*
  Colors.
*/
float orange[4] = { 1.0f, 0.4f,   0.0f,   0.0f };
float green[4]  = { 0.0f, 1.0f,   0.0f,   0.0f };
float blue[4]   = { 0.0f, 0.598f, 0.797f, 0.0f };
float white[4]  = { 1.0f, 1.0f,   1.0f,   0.0f };

/*
  Rotate ground plane collision model 90 about x-axis.
*/
MeMatrix4 groundTransform =
{
    {  1,  0,  0,  0},
    {  0,  0, -1,  0},
    {  0,  1,  0,  0},
    {  0,  0,  0,  1}
};

MeMatrix3 ballOrientation = { 1, 0, 0, 0, 1, 0, 0, 0, 1 };
MeMatrix3 boxOrientation = { 1, 0, 0, 0, 1, 0, 0, 0, 1 };

/*
  User help.
*/

char *help[2] =
{
    "$ACTION2: reset",
    "$ACTION3: shoot"
};

/*
  Shoot the ball.
*/
void MEAPI shoot(RRender * rc, void *userData)
{
    MdtBodyEnable(ball);
    MdtBodyEnable(box);
    MdtBodySetPosition(ball, 0, 4, 0);
    MdtBodySetLinearVelocity(ball, 5.0f, -4.0f, 0.0);
    MdtBodySetAngularVelocity(ball, 1, 1, 1);
}

/*
  Reset the scene.
*/
void MEAPI reset(RRender * rc, void *userData)
{
    MdtBodyEnable(ball);
    MdtBodySetPosition(ball, 0, 4, 0);
    MdtBodySetLinearVelocity(ball, 0, 0, 0);
    MdtBodySetAngularVelocity(ball, 0, 0, 0);
    MdtBodySetOrientation(ball, ballOrientation);

    MdtBodyEnable(box);
    MdtBodySetPosition(box, wallPos[0], wallPos[1], wallPos[2]);
    MdtBodySetOrientation(box, boxOrientation);
    MdtBodySetLinearVelocity(box, 0, 0, 0);
    MdtBodySetAngularVelocity(box, 0, 0, 0);
}


/*
  This function is called every frame by the renderer.
*/
void MEAPI tick(RRender * rc, void *userData)
{
    McdSpaceUpdateAll(space);
    MstBridgeUpdateContacts(bridge,space,world);
    MdtWorldStep(world,step*4);
}

void MEAPI_CDECL cleanup(void)
{
    /* free graphics */
    RRenderContextDestroy(rc);

    /* free dynamics */
    MdtWorldDestroy(world);

    /* free collision */

    MstBridgeDestroy(bridge);
    McdSpaceDestroy(space);
    McdFrameworkDestroyAllModelsAndGeometries(frame);
    McdTerm(frame);
}

/*
  Main routine.
*/
int MEAPI_CDECL main(int argc, const char **argv)
{
    MeCommandLineOptions* options;
    /*
      Kea contatct properties.
    */
    MdtContactParamsID params;

    /*
      Initialize renderer
    */
    options = MeCommandLineOptionsCreate(argc, argv);
    rc = RRenderContextCreate(options, 0, !MEFALSE);
    MeCommandLineOptionsDestroy(options);
    if (!rc)
      return 1;

    /*
     * Create a dynamics world with a maximum of 10 bodies,
     * 10 constraints and 1 material. Also pass in the memory area and
     * its size.
     */
    world = MdtWorldCreate(2, 20);
    MdtWorldSetGravity(world, gravity[0], gravity[1], gravity[2]);

    /* disable the body if it comes to rest */
    MdtWorldSetAutoDisable(world, 1);

    /*
      Initialise Mcd system.
    */

    frame = McdInit(0, 100,0);
    McdPrimitivesRegisterTypes(frame);
    McdPrimitivesRegisterInteractions(frame);

    space = McdSpaceAxisSortCreate(frame,McdAllAxes,
        MAX_BODIES, 2 * MAX_BODIES,1);
    bridge = MstBridgeCreate(frame,10);
    MstSetWorldHandlers(world);

    /*
      Ground plane.
    */
    /* ground graphic */
    planeG = RGraphicGroundPlaneCreate(rc, 30.0f,30, white, 0);
    RGraphicSetTexture(rc, planeG, "checkerboard");

    /* ground collision */
    plane_prim = McdPlaneCreate(frame);
    groundCM = McdModelCreate(plane_prim);
    McdSpaceInsertModel(space, groundCM);

    /* orient the collision body */
    McdModelSetTransformPtr(groundCM, groundTransform);

    /* inform the system that this collision body
       will never move */
    McdSpaceUpdateModel(groundCM);
    McdSpaceFreezeModel(groundCM);

    /*
      Wall.
    */
    /* create wall dynamics */
    box = MdtBodyCreate(world);
    MdtBodyEnable(box);
    MdtBodySetPosition(box, wallPos[0], wallPos[1], wallPos[2]);
    MdtBodySetMass(box, boxMass);

    /* create wall graphic */
    boxG = RGraphicBoxCreate(rc, wallDims[0],  wallDims[1],
        wallDims[2], blue, MdtBodyGetTransformPtr(box));

    /* create wall collision information */
    box_prim1 = McdBoxCreate(frame,wallDims[0], wallDims[1], wallDims[2]);
    boxCM = McdModelCreate(box_prim1);
    McdSpaceInsertModel(space, boxCM);
    McdModelSetBody(boxCM, box);

    /*
      Ball.
    */
    /* ball dynamics */
    ball = MdtBodyCreate(world);
    MdtBodyEnable(ball);
    MdtBodySetPosition(ball, 0, 4, 0);
    MdtBodySetMass(ball, ballMass);

#ifdef USE_CUBE
    /* cube graphic */
    ballG = RGraphicBoxCreate(rc, 2*radius, 2*radius, 2*radius, orange,
        MdtBodyGetTransformPtr(ball));

    /* cube collision */
    box_prim = McdBoxCreate(2*radius, 2*radius, 2*radius);
    ballCM = McdModelCreate(box_prim);
    McdSpaceInsertModel(space, ballCM);
    McdModelSetBody(ballCM, ball);
#else
    /* Sphere graphic */
    ballG = RGraphicSphereCreate(rc, radius, orange, MdtBodyGetTransformPtr(ball));
    RGraphicSetTexture(rc, ballG, "ME_ball3");

    /* sphere collision */
    ball_prim = McdSphereCreate(frame,radius);
    ballCM = McdModelCreate(ball_prim);
    McdSpaceInsertModel(space, ballCM);
    McdModelSetBody(ballCM, ball);
#endif

#ifdef USE_HINGE
    /* Hinge joint */
    hinge = MdtHingeCreate(world);
    MdtHingeSetBodies(hinge,box,0);
    MdtHingeSetPosition(hinge, wallPos[0], wallPos[1],
        wallPos[2] + wallDims[2]/2);
    MdtHingeSetAxis(hinge, 0, 1, 0);
    MdtHingeEnable(hinge);
#endif

    /*
      Set parameters for contacts.
    */
    params = MstBridgeGetContactParams(bridge,
        MstBridgeGetDefaultMaterial(), MstBridgeGetDefaultMaterial());

    MdtContactParamsSetType(params, MdtContactTypeFriction2D);

    /*
      Build space.
    */
    McdSpaceBuild(space);

    /*
      Camera position.
    */

    RCameraSetView(rc,(float)15,(float)-0.1,(float)0.5);

    /*
      Keyboard callbacks.
    */

    RRenderSetActionNCallBack(rc, 2, reset, 0);
    RRenderSetActionNCallBack(rc, 3, shoot, 0);

    /*
      On screen help text.
    */
    RRenderSetWindowTitle(rc,"BallHitsWall3 tutorial");
    RRenderCreateUserHelp(rc, help, 2);
    RRenderToggleUserHelp(rc);

    /*
      Cleanup after simulation.
    */
#ifndef PS2
    atexit(cleanup);
#endif
    /*
      Run the simulation loop.
    */
    RRun(rc, tick,0);

    return 0;
}
